/***********************************************************************************
Project:          SHT11 demo program (V2.0)
Filename:         SHT11.c    

Prozessor:        80C51 family
Compiler:         Keil Version 6.14

Autor:            MST
Copyrigth:        (c) Sensirion AG      
***********************************************************************************/

#include <reg167.h>  //Microcontroller specific library, e.g. port definitions
#include <intrins.h> //Keil library (is used for _nop()_ operation)  
#include <math.h>    //Keil library  
#include <stdio.h>   //Keil library


//----------------------------------------------------------------------------------
// modul-var
//----------------------------------------------------------------------------------
enum {TEMP,HUMI};

#define noACK 0
#define ACK   1
                            //adr  command  r/w
#define STATUS_REG_W 0x06   //000   0011    0
#define STATUS_REG_R 0x07   //000   0011    1
#define MEASURE_TEMP 0x03   //000   0001    1
#define MEASURE_HUMI 0x05   //000   0010    1
#define RESET        0x1e   //000   1111    0

sbit P310 = P3^10; 	// Namenvergabe fuer Portpins UART	
sbit DP310 = DP3^10;// Namenvergabe fuer Richtung UART

sbit DATA = P4^4;
sbit SCK = P4^3;
sbit DDATA = DP4^4;
sbit DSCK = DP4^3;

unsigned char lsb,msb;
unsigned int wert,checksum;

//---------------------------------------------------------------------------------
//  Delay(); kleine Warteschleife
//  Bleibt in der Schleife bis 'count' null wird 
//  Input:	        1 count = 400ns
//  Return:			None
//---------------------------------------------------------------------------------
void Delay(unsigned char count)
{
   while (count--);
}



//----------------------------------------------------------------------------------
char s_write_byte(unsigned char value)
//----------------------------------------------------------------------------------
// writes a byte on the Sensibus and checks the acknowledge 
{ 
  unsigned char i,err=0;  
  DDATA=1;							//DATA on output
  for (i=0x80;i>0;i/=2)             //shift bit for masking
  { if (i & value) DATA=1;          //masking value with i , write to SENSI-BUS
    else DATA=0;                        
    SCK=1;                          //clk for SENSI-BUS
    Delay(13);        				//pulswith approx. 5 us  	
    SCK=0;
    }

  DDATA=0;							//DATA on input, release DATA-line
  SCK=1;                            //clk #9 for ack 
  err=DATA;						//check ack (DATA will be pulled down by SHT11)
  SCK=0;
  if (err) printf("error_write_byte");
  return err;                     //err=1 in case of no acknowledge
}

//----------------------------------------------------------------------------------
char s_read_byte(unsigned char ack)
//----------------------------------------------------------------------------------
// reads a byte form the Sensibus and gives an acknowledge in case of "ack=1" 
{ 
  unsigned char i,val=0;

  DDATA=0;							//DATA on input, release DATA-line
  for (i=0x80;i>0;i/=2)             //shift bit for masking
  { 
    SCK=1;                          //clk for SENSI-BUS
    if (DATA) val=(val | i);        //read bit  
    SCK=0;  					 
  }
  if (ack)
	{
	DDATA=1;						//DATA on output
	DATA=0; 	                    //in case of "ack==1" pull down DATA-Line
	}
  SCK=1;                            //clk #9 for ack
  Delay(13);				        //pulswith approx. 5 us 
  SCK=0;						    
  DDATA=0;							//DATA on input, release DATA-line
  return val;
}

//----------------------------------------------------------------------------------
void s_transstart(void)
//----------------------------------------------------------------------------------
// generates a transmission start 
//       _____         ________
// DATA:      |_______|
//           ___     ___
// SCK : ___|   |___|   |______
{  
   DDATA=1;							//DATA on output
   DATA=1; SCK=0;                   //Initial state
   Delay(4);
   SCK=1;
   Delay(4);
   DATA=0;
   Delay(4);
   SCK=0;  
   Delay(13);
   SCK=1;
   Delay(4);
   DATA=1;		   
   Delay(4);
   SCK=0;		   
}

//----------------------------------------------------------------------------------
void s_connectionreset(void)
//----------------------------------------------------------------------------------
// communication reset: DATA-line=1 and at least 9 SCK cycles followed by transstart
//       _____________________________________________________         ________
// DATA:                                                      |_______|
//          _    _    _    _    _    _    _    _    _        ___     ___
// SCK : __| |__| |__| |__| |__| |__| |__| |__| |__| |______|   |___|   |______
{  
  unsigned char i; 
  DDATA=1;							//DATA on output
  DATA=1; SCK=0;                    //Initial state
  for(i=0;i<9;i++)                  //9 SCK cycles
  { 
	SCK=1;
    SCK=0;
  }
  s_transstart();                   //transmission start
}

//----------------------------------------------------------------------------------
char s_softreset(void)
//----------------------------------------------------------------------------------
// resets the sensor by a softreset 
{ 
  unsigned char err=0;  
  s_connectionreset();              //reset communication
  err+=s_write_byte(RESET);       //send RESET-command to sensor
  if (err) printf("error_softreset");
  return err;                     //err=1 in case of no response form the sensor
}

//----------------------------------------------------------------------------------
char s_read_statusreg(unsigned char *p_value, unsigned char *p_checksum)
//----------------------------------------------------------------------------------
// reads the status register with checksum (8-bit)
{ 
  unsigned char err=0;
  s_transstart();                   //transmission start
  err=s_write_byte(STATUS_REG_R); //send command to sensor
  *p_value=s_read_byte(ACK);        //read status register (8-bit)
  *p_checksum=s_read_byte(noACK);   //read checksum (8-bit)  
  if (err) printf("error_read_statusreg");
  return err;                     //err=1 in case of no response form the sensor
}

//----------------------------------------------------------------------------------
char s_write_statusreg(unsigned char *p_value)
//----------------------------------------------------------------------------------
// writes the status register with checksum (8-bit)
{ 
  unsigned char err=0;
  s_transstart();                   //transmission start
  err+=s_write_byte(STATUS_REG_W);//send command to sensor
  err+=s_write_byte(*p_value);    //send value of status register
  if (err) printf("error_write_statusreg");
  return err;                     //err>=1 in case of no response form the sensor
}
 							   
//----------------------------------------------------------------------------------
char s_measure(unsigned char mode)
//----------------------------------------------------------------------------------
// makes a measurement (humidity/temperature) with checksum
{ 
  unsigned err=0;
  unsigned int i;
  unsigned char check;
  
  s_transstart();                   //transmission start
  switch(mode){                     //send command to sensor
    case TEMP	: err+=s_write_byte(MEASURE_TEMP); break;
    case HUMI	: err+=s_write_byte(MEASURE_HUMI); break;
    default     : break;	 
  }
  DDATA=0;							//DATA on input
  for (i=0;i<65535;i++)
  {
   Delay(4);
   if(DATA==0) break; //wait until sensor has finished the measurement
  }
  if(DATA) err+=1;                		// or timeout (~2 sec.) is reached
  msb=s_read_byte(ACK);					//read the first byte (MSB)
  lsb=s_read_byte(ACK);					//read the second byte (LSB)
  wert=(int)lsb+((int)msb*256);	//Schieben des MSB
  check =s_read_byte(noACK);  			//read checksum
  checksum=(int)check;
  if (err) printf("error_measure");
  return err;
}

//----------------------------------------------------------------------------------
void init_uart()
//----------------------------------------------------------------------------------
//9600 bps @ 16.0 MHz 
{
//----- Initialisation Serial Port ----//
   P310=1;         	// set Port 3.10 output latch TxD 
   DP310=1;        	// set Port 3.10 direction control TxD out      
   S0TIC=0x80;     	// set transmit interrupt flag
   S0BG= 0x33;     	// set baudrate 9600 @ 16 MC CPU clock
   S0CON=0x8011;   	// set serial mode
}

//----------------------------------------------------------------------------------------
void calc_sth11(float *p_humidity ,float *p_temperature)
//----------------------------------------------------------------------------------------
// calculates temperature [C] and humidity [%RH] 
// input :  humi [Ticks] (12 bit) 
//          temp [Ticks] (14 bit)
// output:  humi [%RH]
//          temp [C]
{ const float C1=-4.0;              // for 12 Bit
  const float C2= 0.0405;           // for 12 Bit
  const float C3=-0.0000028;        // for 12 Bit
  const float T1=-0.01;             // for 14 Bit @ 5V
  const float T2=0.00008;           // for 14 Bit @ 5V	

  float rh=*p_humidity;             // rh:      Humidity [Ticks] 12 Bit 
  float t=*p_temperature;           // t:       Temperature [Ticks] 14 Bit
  float rh_lin;                     // rh_lin:  Humidity linear
  float rh_true;                    // rh_true: Temperature compensated humidity
  float t_C;                        // t_C   :  Temperature [C]

  t_C=t*0.01 - 40;                  //calc. temperature from ticks to [C]
  rh_lin=C3*rh*rh + C2*rh + C1;     //calc. humidity from ticks to [%RH]
  rh_true=(t_C-25)*(T1+T2*rh)+rh_lin;   //calc. temperature compensated humidity [%RH]
  if(rh_true>100)rh_true=100;       //cut if the value is outside of
  if(rh_true<0.1)rh_true=0.1;       //the physical possible range

  *p_temperature=t_C;               //return temperature [C]
  *p_humidity=rh_true;              //return humidity[%RH]
}

//--------------------------------------------------------------------
float calc_dewpoint(float h,float t)
//--------------------------------------------------------------------
// calculates dew point
// input:   humidity [%RH], temperature [C]
// output:  dew point [C]
{ float logEx,dew_point;
  logEx=0.66077+7.5*t/(237.3+t)+(log10(h)-2);
  dew_point = (logEx - 0.66077)*237.3/(0.66077+7.5-logEx);
  return dew_point;
}

//----------------------------------------------------------------------------------
void main()
//----------------------------------------------------------------------------------
// sample program that shows how to use SHT11 functions
// 1. connection reset 
// 2. measure humidity [ticks](12 bit) and temperature [ticks](14 bit)
// 3. calculate humidity [%RH] and temperature [C]
// 4. calculate dew point [C]
// 5. print temperature, humidity, dew point  

{ 
  float dew_point;
  unsigned char err;
  unsigned int i;
  unsigned int humi_val_i,temp_val_i;
  float humi_val_f,temp_val_f;

  DDATA=1;							//DATA on output
  DSCK=1;							//DSCK on output
  init_uart();
  s_connectionreset();
  while(1)
  { err=0;
    err+=s_measure(HUMI);  //measure humidity
    humi_val_i=wert;
	printf("test_lsb=%8d\n",lsb);
	printf("test_msb=%8d\n",msb);
	printf("wert_hum=%8d\n",wert);
	err+=s_measure(TEMP);  //measure temperature
    temp_val_i=wert;
	printf("test_lsb_temp=%8d\n",lsb);
	printf("test_msb_temp=%8d\n",msb);
	printf("wert_temp=%8d\n",wert);
	if(err!=0)
    {
	 printf("error\n");
	 s_connectionreset();	//in case of an error: connection reset
	 }                 
    else
    { humi_val_f=(float)humi_val_i;                   //converts integer to float
      temp_val_f=(float)temp_val_i;                   //converts integer to float
      calc_sth11(&humi_val_f,&temp_val_f);            //calculate humidity, temperature
      dew_point=calc_dewpoint(humi_val_f,temp_val_f); //calculate dew point
      printf("temp:%5.1fC humi:%5.1f%% dew point:%5.1fC\n",temp_val_f,humi_val_f,dew_point);
    }
    //----------wait approx. 0.8s to avoid heating up SHTxx------------------------------      
    for (i=0;i<5000;i++) Delay(50000);     //(be sure that the compiler doesn't eliminate this line!)
    //-----------------------------------------------------------------------------------                       
  }
} 

